// proxy.js
const http = require('http');
const https = require('https');
const net = require('net');
const url = require('url');
const { Buffer } = require('buffer');

// Load environment variables (create .env file in production)
require('dotenv').config();

// Valid users array (in production, use a secure database)
const validUsers = [
  { user: process.env.USER1 || 'user1', pass: process.env.PASS1 || 'pass1' },
  { user: process.env.USER2 || 'admin', pass: process.env.PASS2 || 'admin' }
];

// Create proxy server
const proxyServer = http.createServer((req, res) => {
  // 🔴 Health Check Route (No Authentication Required)
  if (req.url === '/health') {
    res.writeHead(200, { 'Content-Type': 'application/json' });
    return res.end(JSON.stringify({
      status: 'healthy',
      uptime: Math.floor(process.uptime()),
      timestamp: new Date().toISOString(),
      version: '1.0.0'
    }));
  }

  // 🔐 Authentication Middleware
  const authHeader = req.headers['proxy-authorization'];
  if (!authHeader || !authHeader.startsWith('Basic ')) {
    res.writeHead(407, { 'Proxy-Authenticate': 'Basic realm="Proxy Server"' });
    return res.end('Proxy authentication required');
  }

  const base64Credentials = authHeader.split(' ')[1];
  const [user, pass] = Buffer.from(base64Credentials, 'base64').toString().split(':');

  if (!validUsers.some(u => u.user === user && u.pass === pass)) {
    res.writeHead(403, { 'Proxy-Authenticate': 'Basic realm="Proxy Server"' });
    return res.end('Invalid credentials');
  }

  // 🌐 Proxy Request Handler
  // Parse target URL
  const targetUrl = req.url.startsWith('http') 
    ? new url.URL(req.url) 
    : new url.URL(`http://${req.headers.host}${req.url}`);

  // Forward request
  const options = {
    hostname: targetUrl.hostname,
    port: targetUrl.port || (targetUrl.protocol === 'https:' ? 443 : 80),
    path: targetUrl.pathname + targetUrl.search,
    method: req.method,
    headers: req.headers
  };

  const proxyReq = (targetUrl.protocol === 'https:' ? https : http).request(options, (proxyRes) => {
    res.writeHead(proxyRes.statusCode, proxyRes.headers);
    proxyRes.pipe(res);
  });

  req.pipe(proxyReq);
});

// 🔐 Handle HTTPS CONNECT method
proxyServer.on('connect', (req, clientSocket, head) => {
  const { hostname } = new url.URL(`https://${req.url}`); 
  
  const serverSocket = net.connect(443, hostname, () => {
    clientSocket.write('HTTP/1.1 200 Connection Established\r\n\r\n');
    serverSocket.pipe(clientSocket);
    clientSocket.pipe(serverSocket);
  });
});

// 🚀 Start server
const PORT = process.env.PORT || 3000;
proxyServer.listen(PORT, () => {
  console.log(`🚀 Proxy server running on port ${PORT}`);
  console.log(`🔧 Health check: http://localhost:${PORT}/health`);
});

// ⏳ Performance optimizations
proxyServer.timeout = 0; // No timeout
proxyServer.headersTimeout = 0;
proxyServer.keepAliveTimeout = 0;